C      *****************************************************************
C      * Subroutine NuTree                                             *
C      * Calculates the remaining amount of nutrients in the soil after*
C      * nutrient uptake by the tree.  The amount of hydrogen ion left *
C      * in the soil is calculated by determining the hydrogen ion     *
C      * exudation required for charge balance.                        *
C      *                                                               *
C      * Variables:                                                    *
C      * CumChg    -Local - The cumulative charge.  Units are          *
C      *                    mol-equivalents.                           *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * HIndex    -Local - The hydrogen ion index.  Unitless.         *
C      * HydrEx    -Output- A vector, of size NLayers, of the hydrogen *
C      *                    ion exudation in each layer.  Input from   *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Units are                *
C      *                    mole-equivalents.                          *
C      *                    (Common block VOut, file VModel.f)         *
C      * L         -Local - Layer index.  Unitless.                    *
C      * KT        -In/Out- A matrix, of size NComp by NLayer, of the  *
C      *                    known total amount for each component, for *
C      *                    each layer.  The total includes the amount *
C      *                    of the component as a free component and as*
C      *                    part of all dependent species.  Units are  *
C      *                    mol.                                       *
C      *                    (Common block VModel, file VModel.f)       *
C      * N         -Local - Nutrient index.  Unitless.                 *
C      * Name      -Local - Used for passing data to WhereIs function. *
C      *                    Unitless.                                  *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * NuTake    -Input - A matrix of size NComp by NLayers,  of the *
C      *                    nutrients taken up by the tree in TREGRO   *
C      *                    for each layer.  The order of nutrients is *
C      *                    found by using the array SpIndex.          *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Units are mol.    *
C      *                    (Common block VModel, file VModel.f)       *
C      * SpIndex   -Input - A vector, of size SpNum, of indices into   *
C      *                    arrays which match each species of         *
C      *                    interest.  SpIndex(i) is an index that when*
C      *                    used in the Labels array will give the     *
C      *                    name of the species amount the tree has    *
C      *                    consumed is NuTake(i).  SpIndex(i) when    *
C      *                    used as the index into SpConc will give the*
C      *                    concentration of this species.  Input from *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VIF, file VModel.f)          *
C      * SpNum     -Input - The number of species of interest.  Input  *
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VIF, file VModel.f)          *
C      * WhereIs   -Fcn   - Returns the index of requested species. If *
C      *                    the species is not found a -1 is returned. *
C      *                    Unitless.                                  *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      * Z         -Input - A vector, of size NComp, of unspeciated    *
C      *                    charge for each component.  Units are      *
C      *                    equivalent charge.                         *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE NUTREE(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER GOAHEAD, HINDEX, L, N, WHEREIS
							REAL*8 CUMCHG
							CHARACTER*8 NAME
							
C      *------------------------------------------*
C      * Indicate that no errors have been found. *
C      *------------------------------------------*
       GOAHEAD = 1
	   
C      *-----------------------------*
C      * Clear the cumulative charge *
C      *-----------------------------*
       CUMCHG = 0.0

C      *-------------------------------------------------------------*
C      * Find the index for hydrogen.  If it is not found, output an *
C      * error message and exit routine.                             *
C      *-------------------------------------------------------------*
       NAME = 'H'
       HINDEX = WHEREIS(NAME)
		     IF (HINDEX.EQ.-1) THEN
			       GOAHEAD = 0
										WRITE(6,*)
										WRITE(6,*) 'Hydrogen ion ("H") index not found'
										WRITE(6,*) 'Routine: NuTree,  File: NuTree.f'
										WRITE(YASOUT,*)
										WRITE(YASOUT,*) 'Hydrogen ion ("H") index not found'
										WRITE(YASOUT,*) 'Routine: NuTree,  File: NuTree.f'
										GOTO 9999
		     ENDIF

C      *--------------------------------------------*
C      * Loop over all tree nutrients in all layers *
C      *--------------------------------------------*
       DO 100 L = 1, NLAYERS
	         DO 200 N = 1, SPNUM

C            *--------------------------------------------------------*
C            * Increment the cumulative charge by the number of mole- *
C            * equivalents of the current nutrient.                   *
C            *--------------------------------------------------------*
             CUMCHG = CUMCHG + NUTAKE(N,L) * Z(SPINDEX(N))

C            *--------------------------------------------------------*
C            * Decrement the known total for the current nutrient by  *
C            * the amount taken up by the tree.                       *
C            *--------------------------------------------------------*
             KT(SPINDEX(N),L) = KT(SPINDEX(N),L) - NUTAKE(N,L)
		  
  200  	  CONTINUE
  
C         *------------------------------------------------------*
C         * Hydrogen ion exudation is calculated by dividing the *
C         * cumulative charge by the charge for hydrogen.        *
C         * Add the exudation to the known total for hydrogen.   *
C         *------------------------------------------------------*
		        IF (Z(HINDEX).NE.0) THEN
             HYDREX(L) = CUMCHG / Z(HINDEX)
             KT(HINDEX,L) = KT(HINDEX,L) + HYDREX(L)
		        ELSE
			          GOAHEAD = 0
													WRITE(6,*)
													WRITE(6,*) 'The unspeciated charge (array Z) is'
													WRITE(6,*) 'zero for hydrogen.  This is '
													WRITE(6,*) 'unacceptable as a denominator'
													WRITE(6,*) 'Routine: NuTree,  File: NuTree.f'
													WRITE(YASOUT,*)
													WRITE(YASOUT,*) 'The unspeciated charge (array Z) is'
													WRITE(YASOUT,*) 'zero for hydrogen.  This is '
													WRITE(YASOUT,*) 'unacceptable as a denominator'
													WRITE(YASOUT,*) 'Routine: NuTree,  File: NuTree.f'
													GOTO 9999
		        ENDIF
  100  CONTINUE
  
C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE			

       RETURN 
       END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
